/*****************************************************************************
 *   pmu.c:  Power Management Unit(PMU) file for NXP LPC13xx 
 *   Family Microprocessors
 *
 *   Copyright(C) 2008, NXP Semiconductor
 *   All rights reserved.
 *
 *   History
 *   2008.07.20  ver 1.00    Preliminary version, first Release
 *
*****************************************************************************/
#include "LPC177x_8x.h"			/* LPC13xx Peripheral Registers */
#include "type.h"
#include "extint.h"
#include "pmu.h"

volatile uint32_t pmu_counter = 0;
volatile uint32_t eint0_counter = 0;

/*****************************************************************************
** Function name:		EINT0_Handler
**
** Descriptions:		external INT handler
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void EINT0_IRQHandler (void) 
{
  LPC_SC->EXTINT = EINT0;		/* clear interrupt */
  eint0_counter++;
  if ( eint0_counter & 0x01 )	/* alternate the LED display */
  {
	LPC_GPIO0->SET = 0x00078000;	/* turn off P0.xx */	
  }
  else
  {	
	LPC_GPIO0->CLR = 0x00078000;	/* turn on P0.xx */
  }
  return;
}

/*****************************************************************************
** Function name:		MinimumSystemInit
**
** Descriptions:		Initialize system to minimum configuration before 
						going to sleep or power down mode.
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void MinimumSystemInit (void)
{
  LPC_SC->SCS       = 0x20;
  while ((LPC_SC->SCS & (1<<6)) == 0);/* Wait for Oscillator to be ready    */

  	 LPC_SC->CCLKSEL = 1;
	 LPC_SC->PLL0CON     &= ~(1<<0);                          /*  Disable the main PLL (PLL0) */
	 LPC_SC->PLL0FEED    = 0xAA;                               /*  Feed */
	 LPC_SC->PLL0FEED    = 0x55;                                 /* Feed */
	 while ((LPC_SC->PLL0STAT & (1<<8)) != 0x00);       /*  Wait for main PLL (PLL0) to shut down */

  LPC_SC->PCLKSEL   = 0x1F;    /* Peripheral Clock Selection         */
  LPC_SC->CCLKSEL   = 0x1F;    /* Setup Clock Divider                */

  switch (LPC_SC->CLKSRCSEL & 0x03) {
    case 0:                           /* Internal RC oscillator => PLL0     */
      SystemCoreClock = 12000000 / (LPC_SC->CCLKSEL & 0x1F);
      break;
    case 1:                           /* Main oscillator => PLL0            */
      SystemCoreClock = 12000000 / (LPC_SC->CCLKSEL & 0x1F);
      break;
  }
  return;
}

/*****************************************************************************
** Function name:		PMU_Init
**
** Descriptions:		Initialize PMU and setup wakeup source.
**						For Sleep and deepsleep, any of the I/O pins can be 
**						used as the wakeup source.
**						For Deep Powerdown, only pin P1.4 can be used as 
**						wakeup source from deep powerdown. 
**
** parameters:			None
** Returned value:		None
** 
*****************************************************************************/
void PMU_Init( void )
{
  LPC_IOCON->P2_10 &= ~0x07;
  LPC_IOCON->P2_10 |= 0x01; 		/* P2.10 as EINT0 */
  LPC_IOCON->P2_10 &= ~0x18;		/* Clear PU/PD */
  LPC_IOCON->P2_10 |= 0x10;			/* Set PU mode */
  LPC_GPIO0->DIR |= 0x00078000;		/* port 0, bit 0~4 only */
  LPC_GPIO0->SET = 0x00078000;		/* turn off LEDs */
#if 1
  LPC_GPIOINT->IO2IntEnF = 1<<10;	/* Port2.10 is falling edge. */
#endif
  LPC_SC->EXTINT = EINT0;			/* clear interrupt */
  LPC_SC->EXTMODE = EINT0_EDGE;		/* INT0 edge trigger */
  LPC_SC->EXTPOLAR = 0;				/* INT0 is falling edge by default */

  NVIC_EnableIRQ(EINT0_IRQn);
  return;
}

extern int BenchDump(void);
/*****************************************************************************
** Function name:		PMU_Sleep
**
** Descriptions:		Put some of the peripheral in sleep mode.
**
** parameters:			SleepMode: 1 is deep sleep, 0 is sleep.
** Returned value:		None
** 
*****************************************************************************/
int PMU_Sleep( uint32_t SleepMode )
{	
  LPC_SC->PCON |= 0xF00; // Clear power mode "success" bits
  /* If normal sleep, not deep sleep, don't do anything to SCR reg. */
  if ( SleepMode )
  {
	SCB->SCR |= NVIC_LP_SLEEPDEEP;
  } else
  {
  	SCB->SCR &= NVIC_LP_SLEEPDEEP;
  }
  LPC_SC->PCON &= ~0x03; // Clear Power dowm and deep power down modes

  BenchDump();
  __WFI();
  if(SleepMode)
  {
  	if(LPC_SC->PCON & (1<<9))
		return 1; // successfully entered deep sleep mode
	return 0; // failure
  } else
  {
  	if(LPC_SC->PCON & (1<<8))
		return 1; // successfully entered sleep mode
	return 0; // failure
  }
}

/*****************************************************************************
** Function name:		PMU_PowerDown
**
** Descriptions:		Put MCU in either Powerdown or Deep Powerdown
**
** parameters:			PowerdownMode: 1 is deep powerdown, 0 is powerdown, 
** Returned value:		None
** 
*****************************************************************************/
int PMU_PowerDown( uint32_t PowerdownMode )
{
  LPC_SC->PCON |= 0xF00; // Clear power mode "success" bits
  if ( PowerdownMode )
  {
	SCB->SCR |= NVIC_LP_SLEEPDEEP;
    LPC_SC->PCON |= 0x03;
  }
  else
  {
	SCB->SCR |= NVIC_LP_SLEEPDEEP;
    LPC_SC->PCON &= ~0x03;
    LPC_SC->PCON |= 0x01;
  }
  BenchDump();
  __WFI();
  if(PowerdownMode)
  {
  	if(LPC_SC->PCON & (1<<11))
		return 1; // successfully entered deep powerdown mode
	return 0; // failure
  } else
  {
  	if(LPC_SC->PCON & (1<<10))
		return 1; // successfully entered powerdown mode
	return 0; // failure
  }
}

/******************************************************************************
**                            End Of File
******************************************************************************/
